<?php

namespace App\Http\Controllers;
use App\Bed;
use App\Room;
use App\Station;
use App\Hospital;
use Illuminate\Http\Request;

class RoomController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index($hospital_id, $station_id, Request $request)
    {
        $data = [
            'rooms'     => Station::with('room')->where('id', $station_id)->get(),
            'beds'       => Station::with('bed')->where('id', $station_id)->get(),
            'hospital'  => Hospital::find(1)->where('id', $hospital_id)->first(),
            'hospitals' => Hospital::with('station')->orderBy('created_at', 'desc')->get()
        ];

        if(isset($request->ref_no)) {
            $data['ref_no'] = $request->ref_no;
        }

        if(isset($request->transfer)) {
            $data['patient_id'] = $request->id;
        }

        return view('hospitals.room.index', $data);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Hospital $hospital, Station $station, Request $request)
    {
        $data = [
            'hospital'  => $hospital,
            'station'   => $station,
            'rooms'     => Room::where('station_id', $station->id)->get()
        ];

        if(isset($request->id)) {
            $data['room'] = Room::where('id', $request->id)->first();
        }

        return view('hospitals.room.create', $data);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'hospital_id' => 'required',
            'station_id' => 'required',
            'room' => 'required|max:255'
        ]);

        Room::create(request()->all());

        $hospital = Hospital::select('hospital')->where('id', $request->hospital_id)->first();
        $station = Station::select('station')->where('id', $request->station_id)->first();

        AuditTrailController::store(
            'room',
            'create',
            'Create room, '. $hospital->hospital. ' - '. $station->station. ' - ' .$request->room
        );

        return redirect()->back()->with('room.store', true);
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Room  $room
     * @return \Illuminate\Http\Response
     */
    public function show(Room $room)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Room  $room
     * @return \Illuminate\Http\Response
     */
    public function edit(Room $room)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Room  $room
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Room $room)
    {
        $room = Room::where('id', $request->id)->first();
        $room->room = $request->room;
        $room->description = $request->description;
        $room->save();
        return redirect('hospital/'.$room->hospital_id.'/station/'.$room->station_id.'/room/create')->with('room.update', true);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Room  $room
     * @return \Illuminate\Http\Response
     */
    public function destroy(Room $room)
    {
        $bed = Bed::where('room_id', $room->id)->where('status', '!=', 1)->get();
        if(count($bed) > 0) {
            AuditTrailController::store(
                'rooms',
                'delete',
                'Attempting to delete room ['.$room->room.'] with occupied bed | time ['.now().']'
            );
            return redirect()->back();
        }
        Room::destroy($room->id);
        Bed::where('room_id', $room->id)->delete();

        $hospital = Hospital::select('hospital')->where('id', $room->hospital_id)->first();
        $station = Station::select('station')->where('id', $room->station_id)->first();

        AuditTrailController::store(
            'room',
            'delete',
            'Delete room, '. $hospital->hospital. ' - '. $station->station. ' - ' .$room->room
        );

        return redirect()->back()->with('room.destroy', true);
    }
}
