<?php

namespace App\Http\Controllers;

use App\AssignedUser;
use App\Hospital;
use App\Station;
use App\Room;
use App\Bed;
use App\Referral;
use App\Patient;
use App\DepartmentBed;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class PatientController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $hospital_id = Auth::user()->hospital_id;

        $data = [
            'patients'  =>  Referral::where('ref_to_hospital', $hospital_id)
                ->where('is_internal', true)
                ->orderBy('created_at', 'desc')
                ->paginate(20)
        ];

        // dd($data);

        return view('patient.index', $data);
    }

    public function search(Request $request)
    {
        if (!$request->_token) return response()->json(array());

        $patient = Referral::where('patient', 'like', '%' . $request->search . '%')
            ->where('is_internal', true)
            ->where('ref_to_hospital', Auth::user()->hospital_id)
            ->limit(20)->get();

        return response()->json($patient);
    }

    public function admitted()
    {
        $hospital_id = Auth::user()->hospital_id;

        $data = [
            'hospitals' => Hospital::with('station')->orderBy('created_at', 'desc')->get(),
            'patients'  => Patient::select('patients.*', 'referrals.patient', 'referrals.age', 'referrals.created_at as ref_created', 'referrals.referral_no')
                ->join('referrals', 'referrals.id', '=', 'patients.referral_id')
                ->where([
                    ['patients.hospital_id', $hospital_id],
                    ['patients.status', 3]
                ])
                ->orderBy('patients.created_at', 'desc')
                ->get(),
        ];

        return view('patient.index', $data);
    }

    public function released()
    {
        $hospital_id = Auth::user()->hospital_id;

        $data = [
            'hospitals' => Hospital::with('station')->orderBy('created_at', 'desc')->get(),
            'patients'  => Patient::select('patients.*', 'referrals.patient', 'referrals.age', 'referrals.created_at as ref_created', 'referrals.referral_no')
                ->join('referrals', 'referrals.id', '=', 'patients.referral_id')
                ->where([
                    ['patients.hospital_id', $hospital_id],
                    ['patients.status', 4]
                ])
                ->orderBy('patients.created_at', 'desc')
                ->get(),
        ];

        return view('patient.index', $data);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'patient'      => 'required',
            'age'               => 'required',
            'sex'               => 'required',
            'birthdate'        => 'required',
            'address'           => 'required',
            'guardian'          => 'required',
            'complaint'         => 'required',
            'diagnosis'         => 'required',
            'reason'            => 'required',
            'ref_by'            => 'required',
            'contact_no'        => 'required',
            'referral_type'     =>  'required'
        ]);

        if (isset($request->bed_id)) {
            $bed = Bed::where('id', $request->bed_id)->first();

            $hospital_code = Hospital::select('code')->where('id', '=', $bed->hospital_id)->first();
            $ref_ctr = Referral::latest('ref_ctr')->first();
            $ref_ctr = $ref_ctr != null ? strval($ref_ctr->ref_ctr + 1) : '1';
            $ref_len = 6 - strlen($ref_ctr);
            $ref_zero = '';
            for ($i = 0; $i < $ref_len; $i++) {
                $ref_zero .= '0';
            }
            $ref_no = $hospital_code->code . '-' . $ref_zero . $ref_ctr;

            $data = request()->all();

            if ($request->vaccine_1st_dose == 'on') {
                $data['vaccine_1st_dose'] = true;
            }
            if ($request->vaccine_2nd_dose == 'on') {
                $data['vaccine_2nd_dose'] = true;
            }
            if ($request->rtpcr == '+') {
                $data['rtpcr'] = true;
            } elseif ($request->rtpcr == '-') {
                $data['rtpcr'] = false;
            } else {
                $data['rtpcr'] = null;
            }

            if ($request->rat == '+') {
                $data['rat'] = true;
            } elseif ($request->rat == '-') {
                $data['rat'] = false;
            } else {
                $data['rat'] = null;
            }

            $data['referral_no'] = $ref_no;
            $data['ref_ctr'] = intval($ref_ctr);
            $data['ref_from_date'] = now();
            $data['ref_from_time'] = now();
            $data['created_by'] = Auth::user()->id;
            $data['ref_to_hospital'] = $bed->hospital_id;
            $data['ref_status'] = 3;
            $data['is_internal'] = true;
            // dd($data);

            if ($request->rtpcr_attach !== null) {
                $fileName = time() . '-' . $ref_no . '.' . $request->rtpcr_attach->extension();
                $request->rtpcr_attach->move(public_path('rtpcr-attachments'), $fileName, 0777);
                $data['rtpcr_attach'] = $fileName;
            }

            if ($request->rat_attach !== null) {
                $fileName = time() . '-' . $ref_no . '.' . $request->rat_attach->extension();
                $request->rat_attach->move(public_path('rat-attachments'), $fileName, 0777);
                $data['rat_attach'] = $fileName;
            }

            $referral = Referral::create($data);

            // $referral = Referral::create(array_merge(request()->all(), ['referral_no'=>$ref_no, 'ref_ctr'=>intval($ref_ctr), 'ref_to_hospital'=>$bed->hospital_id, 'ref_from_date'=>now(), 'ref_from_time'=>now(), 'created_by'=>Auth::user()->id, 'ref_status'=>3, 'is_internal'=>true]));

            $patient = array(
                'referral_id'   => $referral->id,
                'status'        =>  3,
                'hospital_id'   => $bed->hospital_id,
                'station_id'    => $bed->station_id,
                'room_id'       => $bed->room_id,
                'bed_id'        => $bed->id,
                'bed_no'        => $bed->bed
            );
            $patient = Patient::create($patient);

            $bed->status = 3;
            $bed->patient_id = $patient->id;
            $bed->ref_no = $referral->referral_no;
            $bed->save();

            return redirect('/hospital/' . $bed->hospital_id)->with('walkin', true);
        } elseif (isset($request->dept_bed_id)) {
            $dept_bed = DepartmentBed::where('id', $request->dept_bed_id)->first();

            $hospital_code = Hospital::select('code')->where('id', '=', $dept_bed->hospital_id)->first();
            $ref_ctr = Referral::latest('ref_ctr')->first();
            $ref_ctr = $ref_ctr != null ? strval($ref_ctr->ref_ctr + 1) : '1';
            $ref_len = 6 - strlen($ref_ctr);
            $ref_zero = '';
            for ($i = 0; $i < $ref_len; $i++) {
                $ref_zero .= '0';
            }
            $ref_no = $hospital_code->code . '-' . $ref_zero . $ref_ctr;

            $data = request()->all();

            if ($request->vaccine_1st_dose == 'on') {
                $data['vaccine_1st_dose'] = true;
            }
            if ($request->vaccine_2nd_dose == 'on') {
                $data['vaccine_2nd_dose'] = true;
            }
            if ($request->rtpcr == '+') {
                $data['rtpcr'] = true;
            } elseif ($request->rtpcr == '-') {
                $data['rtpcr'] = false;
            } else {
                $data['rtpcr'] = null;
            }

            if ($request->rat == '+') {
                $data['rat'] = true;
            } elseif ($request->rat == '-') {
                $data['rat'] = false;
            } else {
                $data['rat'] = null;
            }

            $data['referral_no'] = $ref_no;
            $data['ref_ctr'] = intval($ref_ctr);
            $data['ref_from_date'] = now();
            $data['ref_from_time'] = now();
            $data['created_by'] = Auth::user()->id;
            $data['ref_to_hospital'] = $dept_bed->hospital_id;
            $data['ref_status'] = 3;
            $data['is_internal'] = true;
            $data['is_dept_bed'] = true;
            // dd($data);

            $referral = Referral::create($data);

            // $referral = Referral::create(array_merge(request()->all(), ['referral_no'=>$ref_no, 'ref_ctr'=>intval($ref_ctr), 'ref_to_hospital'=>$dept_bed->hospital_id, 'ref_from_date'=>now(), 'ref_from_time'=>now(), 'created_by'=>Auth::user()->id, 'ref_status'=>3, 'is_dept_bed'=>true, 'is_internal'=>true]));

            $dept_bed->ref_id = $referral->id;
            $dept_bed->ref_no = $referral->referral_no;
            $dept_bed->status = 3;
            $dept_bed->save();
            return redirect('/hospital/' . $dept_bed->hospital_id)->with('walkin', true);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Patient  $patient
     * @return \Illuminate\Http\Response
     */
    public function show(Patient $patient)
    {
        $data = [
            'hospitals' => Hospital::with('station')->orderBy('created_at', 'desc')->get(),
        ];

        return redirect('/referral/' . $patient->referral_id);
    }

    public function new_assesment(Referral $referral)
    {
        $data = [
            'hospitals' => Hospital::with('station')->orderBy('created_at', 'desc')->get(),
            'referral'  => $referral
        ];
        $patient = Patient::where('referral_id', $referral->id)->first();
        if ($patient != null) {
            $data['patient']    = $patient;
            $data['bed']        = Bed::where('id', $patient->bed_id)->first();
            $data['room']       = Room::where('id', $patient->room_id)->first();
            $data['station']    = Station::where('id', $patient->station_id)->first();
        }

        $bed = Bed::where('hospital_id', $referral->ref_to_hospital)->first();
        return view('patient.create', $data);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Patient  $patient
     * @return \Illuminate\Http\Response
     */
    public function edit(Patient $patient)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Patient  $patient
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request)
    {
        $patient = Patient::where('id', $request->patient_id)->first();

        //update old Bed into available
        Bed::where('id', $patient->bed_id)->update([
            'status'        =>  1,
            'patient_id'    =>  null,
            'ref_no'        => null
        ]);

        $referral = Referral::where('id', $patient->referral_id)->first();

        //get selected Bed
        $bed = Bed::where('id', $request->bed_id)->first();
        $bed->patient_id = $request->patient_id;
        $bed->status = $patient->status;
        $bed->ref_no = $referral->referral_no;
        $bed->save();

        $patient->station_id = $bed->station_id;
        $patient->room_id = $bed->room_id;
        $patient->bed_id = $bed->id;
        $patient->bed_no = $bed->bed;
        $patient->save();

        return redirect('/referral/' . $patient->referral_id);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Patient  $patient
     * @return \Illuminate\Http\Response
     */
    public function destroy(Patient $patient)
    {
        //
    }
}
