<?php

namespace App\Http\Controllers;

use App\AssignedUser;
use App\Hospital;
use App\Station;
use App\Room;
use App\Bed;
use App\Referral;
use App\Ambulance;
use App\Doctor;
use App\DepartmentBed;
use App\DoctorSchedule;
use App\Notification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

use Carbon\Carbon;

class HospitalController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        // $this->middleware('hod.superadmin');
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $data = [
            'hospitals'     => Hospital::with('station')->orderBy('created_at', 'desc')->get(),
            // 'notifs'        => Notification::where([ ['receiver_role', 2], ['is_seen', false] ])->get()
        ];

        if (Auth::user()->role == 2) {
            $data['notifs'] = Notification::where([['receiver_role', 2], ['is_seen', false]])->get();
        }

        return view('hospitals.index', $data);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {

        $data = [
            'hospitals'     => Hospital::with('station')->orderBy('created_at', 'desc')->get()
        ];
        return view('hospitals.create', $data);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'hospital' => 'required|unique:hospitals|max:255',
            'address' => 'required',
        ]);

        $hospital = new Hospital();
        $hospital->hospital = $request->hospital;
        $hospital->address = $request->address;
        $hospital->created_by = Auth::user()->id;

        $code = '';
        $h = explode(" ", $request->hospital);
        foreach ($h as $i) {
            $code .= $i[0];
        }
        $hospital->code = strtoupper($code);
        $hospital->save();

        AuditTrailController::store(
            'hospital',
            'create',
            'Create hospital, ' . $request->hospital
        );

        return redirect()->back()->with('hospital.store', true);
    }


    /**
     * Display the specified resource.
     *
     * @param  \App\Hospital  $hospital
     * @return \Illuminate\Http\Response
     */
    public function show(Hospital $hospital, Request $request)
    {
        $role = Auth::user()->role;

        if ($role == 4) {
            return redirect()->back();
        }

        if ($role == 1 && Auth::user()->hospital_id != $hospital->id) {
            return redirect()->back();
        }

        $today = Carbon::today();

        $data = [
            'hospital'  => $hospital,
            'station'   => Hospital::with('station')->where('id', $hospital->id)->get(),
            'room'      => Hospital::with('room')->where('id', $hospital->id)->get(),
            'bed'       => Hospital::with('bed')->where('id', $hospital->id)->get(),
            'beds'      => Bed::where('beds.hospital_id', $hospital->id)
                ->select('beds.id', 'beds.ref_no', 'beds.bed', 'beds.status', 'beds.patient_id', 'beds.hospital_id', 'beds.station_id', 'beds.room_id', 'stations.station', 'rooms.room')
                ->join('stations', 'stations.id', '=', 'beds.station_id')
                ->join('rooms', 'rooms.id', '=', 'beds.room_id')
                ->get(),
            'stations'  => Station::where('hospital_id', $hospital->id)->get(),
            'rooms'     => Room::where('hospital_id', $hospital->id)->get(),
            'ambulances' => Ambulance::where('hospital_id', $hospital->id)->get(),
            'doctors'   => Doctor::where('hospital_id', $hospital->id)->get(),
            'or_beds'   => DepartmentBed::where([
                ['hospital_id', $hospital->id],
                ['type', 'or']
            ])->get(),
            'er_beds'        => DepartmentBed::where([
                ['hospital_id', $hospital->id],
                ['type', 'er']
            ])->get(),
            'dr_beds'        => DepartmentBed::where([
                ['hospital_id', $hospital->id],
                ['type', 'dr']
            ])->get(),
            'opd_beds'        => DepartmentBed::where([
                ['hospital_id', $hospital->id],
                ['type', 'opd']
            ])->get(),
            'er_doctors' => DoctorSchedule::select('doctor_schedules.*', 'doctors.doctor', 'doctors.specialty')
                ->join('doctors', 'doctors.id', '=', 'doctor_schedules.doctor_id')
                ->where([
                    ['doctors.hospital_id', $hospital->id],
                    ['doctor_schedules.department', 'er']
                ])
                ->whereDate('doctor_schedules.day', '=', $today->toDateString())
                ->get(),
            'dr_doctors' => DoctorSchedule::select('doctor_schedules.*', 'doctors.doctor', 'doctors.specialty')
                ->join('doctors', 'doctors.id', '=', 'doctor_schedules.doctor_id')
                ->where([
                    ['doctors.hospital_id', $hospital->id],
                    ['doctor_schedules.department', 'dr']
                ])
                ->whereDate('doctor_schedules.day', '=', $today->toDateString())
                ->get(),
            'or_doctors' => DoctorSchedule::select('doctor_schedules.*', 'doctors.doctor', 'doctors.specialty')
                ->join('doctors', 'doctors.id', '=', 'doctor_schedules.doctor_id')
                ->where([
                    ['doctors.hospital_id', $hospital->id],
                    ['doctor_schedules.department', 'or']
                ])
                ->whereDate('doctor_schedules.day', '=', $today->toDateString())
                ->get(),
            'opd_doctors' => DoctorSchedule::select('doctor_schedules.*', 'doctors.doctor', 'doctors.specialty')
                ->join('doctors', 'doctors.id', '=', 'doctor_schedules.doctor_id')
                ->where([
                    ['doctors.hospital_id', $hospital->id],
                    ['doctor_schedules.department', 'opd']
                ])
                ->whereDate('doctor_schedules.day', '=', $today->toDateString())
                ->get()

        ];

        if ($role == 1) {
            $data['notifs'] = Notification::where([['receiver_role', 1], ['is_seen', false]])->get();
        } elseif ($role == 2) {
            $data['notifs'] = Notification::where([['receiver_role', 2], ['is_seen', false]])->get();
        }

        if (isset($request->transfer)) {
            $data['trans_ref_id'] = $request->referral_id;

            $referral = Referral::where('id', $request->referral_id)->first();
            $data['referral_no'] = $referral->referral_no;
        }


        if (Auth::user()->role != 1) {
            return view('hospitals.show', $data);
        } else {
            if (Auth::user()->hospital_id == $hospital->id) {
                return view('hospitals.show', $data);
            } else {
                return redirect('error-404');
            }
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Hospital  $hospital
     * @return \Illuminate\Http\Response
     */
    public function edit(Hospital $hospital)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Hospital  $hospital
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Hospital $hospital)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Hospital  $hospital
     * @return \Illuminate\Http\Response
     */
    public function destroy(Hospital $hospital)
    {
        Hospital::where('id', $hospital->id)->delete();
        return redirect()->back();
    }
}
