<?php

namespace App\Http\Controllers;

use App\Bed;
use App\Room;
use App\Station;
use App\Hospital;
use Illuminate\Http\Request;

class BedController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index($hospital_id, $station_id, $room_id, Request $request)
    {
        $data = [
            'beds'     => Room::with('bed')->where('id', $room_id)->get(),
            'rooms'     => Station::with('room')->where('id', $station_id)->get(),
            'hospital'  => Hospital::find(1)->where('id', $hospital_id)->first(),
            'hospitals' => Hospital::with('station')->orderBy('created_at', 'desc')->get()
        ];

        if(isset($request->ref_no)) {
            $data['ref_no'] = $request->ref_no;
        }

        if(isset($request->transfer)) {
            $data['patient_id'] = $request->id;
        }

        return view('hospitals.bed.index', $data);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Hospital $hospital, Station $station, Room $room, Request $request)
    {
        $data = [
            'hospital'  => $hospital,
            'station'   => $station,
            'room'      => $room,
            'beds'      => Bed::where('room_id', $room->id)->get()
        ];

        if(isset($request->id)) {
            $data['bed'] = Bed::where('id', $request->id)->first();
        }

        return view('hospitals.bed.create', $data);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'hospital_id' => 'required',
            'station_id' => 'required',
            'room_id'   => 'required',
            'bed' => 'required|max:255'
        ]);

        Bed::create(request()->all());

        $hospital = Hospital::select('hospital')->where('id', $request->hospital_id)->first();
        $station = Station::select('station')->where('id', $request->station_id)->first();
        $room = Room::select('room')->where('id', $request->room_id)->first();

        AuditTrailController::store(
            'bed',
            'create',
            'Create bed, '. $hospital->hospital. ' - '. $station->station. ' - ' .$room->room. ' - ' .$request->bed
        );

        return redirect()->back()->with('bed.store', true);
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Bed  $bed
     * @return \Illuminate\Http\Response
     */
    public function show(Bed $bed)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Bed  $bed
     * @return \Illuminate\Http\Response
     */
    public function edit(Bed $bed)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Bed  $bed
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Bed $bed)
    {
        $bed = Bed::where('id', $request->id)->first();
        $bed->bed = $request->bed;
        $bed->description = $request->description;
        $bed->save();

        return redirect('hospital/'.$bed->hospital_id.'/station/'.$bed->station_id.'/room/'.$bed->room_id.'/bed/create')->with('bed.update', true);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Bed  $bed
     * @return \Illuminate\Http\Response
     */
    public function destroy(Bed $bed)
    {
        if($bed->status != 1) {
            AuditTrailController::store(
                'beds',
                'delete',
                'Attempting to delete occupied bed ['.$bed->bed.'] with ref # ['.$bed->ref_no.'] | time ['.now().']'
            );
            return redirect()->back();
        }
        Bed::where('id', $bed->id)->delete();
        return redirect()->back()->with('bed.destroy', true);
    }

    public function unavailable(Request $request)
    {
        Bed::where('id', $request->b_id)->update([
            'status'  =>  3
        ]);
        return redirect()->back();
    }

    public function available(Request $request)
    {
        Bed::where('id', $request->b_id)->update([
            'status'  =>  1
        ]);
        return redirect()->back();
    }
}
